/*
	csv2html
	Written by Dan Gookin, February 24, 2015

	This code takes a command-separated values (CSV) file as input and
	generates an HTML table as output.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TRUE 1
#define FALSE 0
#define SIZE 256

int html;
int header;

enum html_tags
{
	TABLE_START,
	ROW_START,
	CELL_START,
	CELL_END,
	TABLE_HEAD_START,
	TABLE_HEAD_END,
	ROW_END,
	TABLE_END
};

/*
	Display help message and bail
*/
void show_help(void)
{
	puts("csv2html\nWritten by Dan Gookin, 2015\n");
	puts("This filter converts CSV input to HTML table output. Format:\n");
	puts("csv2html [--html] [--header] [--help]\n");
	puts("--html   output full HTML document.");
	puts("--header CSV input contains a header row.");
	puts("--help   display this message\n");
	exit(1);
}

/*
	Parse optional command line arguments
	Doesn't check for duplicates or invalid options
*/
void check_args(int count,char **args)
{
	int x;

	html = header = FALSE;	
	for(x=1;x<count;x++)
	{
		if( strcmp(*(args+x),"--help")==0 )
			show_help();
		if( strcmp(*(args+x),"--html")==0 )
			html=TRUE;
		if( strcmp(*(args+x),"--header")==0 )
			header=TRUE;
	}
}

/*
	Output a generic HTML header
*/
void html_header(void)
{
	puts("<HTML>");
	puts("\t<HEAD>");
	puts("\t\t<TITLE>CSV HTML Table</TITLE>");
	puts("\t\t<META name=\"Author\" content=\"csv2html\">");
	puts("\t</HEAD>");
	puts("\t<BODY>");
}

/*
	Output a generic HTML footer
*/
void html_footer(void)
{
	puts("\t</BODY>");
	puts("</HTML>");
}

/*
	output an HTML table tag
*/
void html_tag_output(int tag)
{
	switch(tag)
	{
		case TABLE_START:
			if(html)				/* indent to align with HTML */
				printf("\t\t");
			puts("<TABLE>");
			break;
		case ROW_START:
			if(html)
				printf("\t\t");
			puts("\t<TR>");
			break;
		case CELL_START:
			if(html)
				printf("\t\t");
			printf("\t\t<TD>");
			break;
		case CELL_END:
			puts("</TD>");
			break;
		case TABLE_HEAD_START:
			if(html)
				printf("\t\t");
			printf("\t\t<TH>");
			break;
		case TABLE_HEAD_END:
			puts("</TH>");
			break;
		case ROW_END:
			if(html)
				printf("\t\t");
			puts("\t</TR>");
			break;
		case TABLE_END:
			if(html)
				printf("\t\t");
			puts("</TABLE>");
			break;
		default:
			break;
	}
}

/*
	Display text stored in the ring buffer
*/
void show_text(char *ring_buffer, char *start)
{
	while(*start)
	{
		putchar(*start);
		start++;
		if(start > ring_buffer+SIZE)
			start = ring_buffer;
	}
}

/*
	Main I/O
*/
int main(int argc, char *argv[])
{
	char ring_buffer[SIZE];
	char *start,*end,ch;
	int row,current_row;

	if(argc > 1)
	{
		check_args(argc,argv);
	}
	
	/* initialize the ring buffer and row counters */
	start = end = ring_buffer;
	row = 1;
	current_row = 0;
	
	/* output HTML header if option set*/
	if(html)
		html_header();
	
	/* output beginning of HTML table */
	html_tag_output(TABLE_START);

	/* Read standard input */
	while( (ch = getchar()) != EOF)
	{
		if(row != current_row)
		{
			html_tag_output(ROW_START);
			if(header && row==1)
				html_tag_output(TABLE_HEAD_START);
			else
				html_tag_output(CELL_START);
			current_row = row;
		}
		switch(ch)
		{
			case 0x0d:			/* carriage return */
								/* if an LF follows the CR, it's dealt with below */
			case 0x0a:			/* line feed */
				*end = '\0';
				show_text(ring_buffer,start);
				start = end;
				if(header && row==1)
					html_tag_output(TABLE_HEAD_END);
				else
					html_tag_output(CELL_END);
				html_tag_output(ROW_END);
				row++;
				break;
			case ',':			/* cell separator */
				*end = '\0';
				show_text(ring_buffer,start);
				start = end;
				if(header && row==1)
				{
					html_tag_output(TABLE_HEAD_END);
					html_tag_output(TABLE_HEAD_START);
				}
				else
				{
					html_tag_output(CELL_END);
					html_tag_output(CELL_START);
				}
				break;
			default:
				if(ch == 0x0a)	/* remove LF after CR */
					break;
				*end++ = ch;
				if(end > ring_buffer+SIZE)
					end = ring_buffer;
		}
	}
	/* check to see whether the ring buffer is empty */
	if(start != end)
	{
		*end = '\0';		/* flush the ring buffer */
		show_text(ring_buffer,start);
		html_tag_output(CELL_END);
		html_tag_output(ROW_END);
	}

	/* output the end of HTML table */
	html_tag_output(TABLE_END);
	
	/* optionally output the rest of HTML */
	if(html)
		html_footer();
	
	return(0);
}
